<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\PatientRequest;
use App\Models\Branch\Appointment;
use App\Models\Branch\Appointment\Rate_appointment;
use App\Models\Branch\Branch;
use App\Models\Branch\Lab;
use App\Models\Branch\Unit;
use App\Models\Invoice\Coupon;
use App\Models\Invoice\Invoice;
use App\Models\Invoice\Invoice_item;
use App\Models\location\City;
use App\Models\location\Country;
use App\Models\Patient\Cancel_reason_cat;
use App\Models\Patient\From_recourse;
use App\Models\Patient\Medicine;
use App\Models\Patient\Patient;
use App\Models\Patient\Service_item;
use App\Models\Patient\Specialty_cat;
use App\Models\Patient\Treatment;
use App\Rules\CouponRule;
use App\Rules\Recaptcha;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class LandController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('landing/index');
    }


    public function login()
    {
        return view('landing/empty/login');
    }

    public function login_sub(Request $request)
    {

        // Validate the form data
        $this->validate($request, [
            'phone_number'   => 'required|numeric',
            'password' => 'required',
            'g-recaptcha-response' => ['required', new Recaptcha()],
        ]);

        // Attempt to log the user in
        if (Auth::guard('patient')->attempt(['phone_number' => $request->phone_number, 'password' => $request->password], $request->remember)) {
            $request->session()->regenerate();
            // if successful, then redirect to their intended location
            return redirect()->intended(route('patient_auth.profile'));
        }

        // if unsuccessful, then redirect back to the login with the form data
        return back()->withErrors([
            'phone_number' => 'Your Phone number or Password is not correct!',
        ]);
    }


    function logout()
    {
        Auth::guard('patient')->logout();
        return redirect()->route('patient_auth.login');
    }


    public function profile()
    {
        $id = Auth::guard('patient')->id();

        $appointments = Appointment::where('patient_id', $id)
            ->with(['branch' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['doctor' => function ($q) {
                $q->select('id', 'first_name');
            }])
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'code', 'status');
                    }]);
            }])
            ->orderBy('id', 'DESC')
            ->get();

        $medicine = Medicine::select('id', 'medicines_cats', 'start', 'end', 'status')->where('patient_id', $id)
            ->with(['medicinescats' => function ($q) {
                $q->select('id', 'name');
            }])
            ->orderBy('status', 'ASC')
            ->get();

        $treatment = Treatment::select('id', 'treatment_cat_id', 'sessions', 'sessions_done', 'start', 'end', 'status')->where('patient_id', $id)
            ->with(['treatment_cat' => function ($q) {
                $q->select('id', 'name');
            }])
            ->orderBy('status', 'ASC')
            ->get();

        $lab = Lab::select('id', 'code', 'services_cat_id', 'resp_doctor_id', 'xray_file', 'status', 'note_lab', 'created_at')->where('patient_id', $id)
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'code', 'status');
                    }]);
            }])
            ->orderBy('status', 'ASC')
            ->get();

        return view('landing/empty/profile', compact('appointments', 'medicine', 'treatment', 'lab'));
    }


    public function register()
    {
        $branches = Branch::all();
        $countries = Country::orderByRaw('RAND()')->get();
        $from_recourses = From_recourse::all();

        return view('landing/empty/register', compact('branches', 'countries', 'from_recourses'));
    }


    //for select input ajax to send the cities beasd on the given country
    public function createcityajax($id)
    {
        return City::where('country_id', $id)->orderBy('fav', 'DESC')->get();
    }

    public function store(PatientRequest $request)
    {

        $this->validate($request, [
            'g-recaptcha-response' => ['required', new Recaptcha()],
        ]);

        //insert img
        if ($request->hasFile('avatar')) {
            $file_extension = request()->avatar->getClientOriginalExtension();
            $file_name = $request->input('first_name') . time() . '.' . $file_extension;
            $path = 'img/useravatar';
            $request->avatar->move($path, $file_name);
        } else {
            $file_name = 'default-pp.png';
        };

        $patient_branch_id = $request->input('first_branch_id');
        $new_serial_number = serial_number('patient');
        $new_serial_number_branch = serial_number('patient', $patient_branch_id);
        $code_branch = Branch::select('id', 'code')->find($patient_branch_id);

        $user = Patient::create([
            'type' => 1,
            'new_id' => $new_serial_number,
            'new_branch_id' => $new_serial_number_branch,
            'code' => $code_branch->code . "-" . $this->generateRandomString(6),
            'first_name' => $request->input('first_name'),
            'second_name' => $request->input('second_name'),
            'email' => $request->input('email'),
            'password' => bcrypt($request->input('password')),
            'first_branch_id' => $request->input('first_branch_id'),
            'avatar' => $file_name,
            'birthday' => $request->input('birthday'),
            'gendar' => $request->input('gendar'),
            'country_id' => $request->input('country_id'),
            'city_id' => $request->input('city_id'),
            'phone_number' => $request->input('phone_number'),
            'sec_phone_number' => $request->input('sec_phone_number'),
            'insurance' => $request->input('insurance'),
            'from_recourse_id' => $request->input('from_recourse_id'),
            'height' => $request->input('height'),
            'weight' => $request->input('weight'),
            'blood_type' => $request->input('blood_type'),
            'note' => $request->input('note'),
        ]);

        Auth::guard('patient')->login($user);

        session()->flash('success', 'The patient has been created successfully');
        return redirect()->route('patient_auth.profile');
    }


    public function appointment()
    {
        $specialties = Specialty_cat::all();
        $branches = Branch::all();
        $services = Service_item::where('service_inv_cat_id', 1)->where('deactivate', 0)->get();

        return view('landing/empty/appointment', compact('specialties', 'branches', 'services'));
    }


    //select the calander data and funcation via ajax in creating
    public function calander_appointment_ajax($month, $year, $specialty_id, $branch_id, $unit_id)
    {

        if (isset(request()->month) && isset(request()->year)) {
            $month = request()->month;
            $year = request()->year;
        } else {
            $dateComponents = getdate();
            $month = $dateComponents['mon'];
            $year = $dateComponents['year'];
        }

        $duration = prox_sett('timeslotduration');
        $cleanup = prox_sett('timeslotcleanup');
        $start = prox_sett('timeslotstart');
        $end = prox_sett('timeslotend');

        if (prox_sett('timeslotweekends') !== "null") {
            $weekends = unserialize(prox_sett('timeslotweekends'));
        } else {
            $weekends = array();
        }

        if ($unit_id !== 'null') {
            return build_calendar($month, $year, $specialty_id, $branch_id, $unit_id, $duration, $cleanup, $start, $end, $weekends);
        } else {
            return "<p class='text-white mb-0'>Sorry, there is no unit related to this branch</p>";
        }
    }


    //fetch servcies besed on speicilty
    public function fetch_servicecat_ajax($specialty_id, $branch_id)
    {
        $service = Service_item::where('service_inv_cat_id', 1)
            ->where('specialty_id', $specialty_id)->where('deactivate', 0);

        if (Auth::user()->branch_id !== 0) {
            $service = $service->whereIn('branch_id', [$branch_id, '0']);
        }

        $service = $service->get();

        return $service;
    }

    //fetch servcies besed on speicilty
    public function land_fetch_unit_ajax($branch_id)
    {

        $unit = Unit::where('branch_id', $branch_id);

        $unit = $unit->get();

        return $unit;
    }


    //select the calander data and funcation via ajax in creating
    public function calander_show_slots_ajax($datetoday, $specialty_id, $branch_id, $unit_id)
    {

        $duration = prox_sett('timeslotduration');
        $cleanup = prox_sett('timeslotcleanup');
        $start = prox_sett('timeslotstart');
        $end = prox_sett('timeslotend');

        return showSlots($duration, $cleanup, $start, $end, $datetoday, $specialty_id, $branch_id, $unit_id);
    }


    public function coupon_search($search_query, $patient_id, $total_price)
    {
        $search_query = request()->search_query;
        $coupon = Coupon::where('code', $search_query)->where('status', '1')->first();

        //if it is empty
        if (!$coupon) {
            $msg = "Sorry! invalid or expired coupon";
            return array('msg' => $msg);
        } else {
            $discount_amount = $coupon->discount($total_price);
            $coupon_id = $coupon->id;

            if ($patient_id === 'null') {
                $msg = "Coupon has been applied";
                $result = array('msg' => $msg, 'discount_amount' => $discount_amount, 'id' => $coupon_id);
                return $result;
            } else {

                $used_coupon = Invoice::where('coupon_id', $coupon_id)->where('receivable_id', $patient_id)->where('receivable_type', 'App\Models\Patient\Patient')->first();

                if (empty($used_coupon)) {
                    $msg = "Coupon has been applied";
                    $result = array('msg' => $msg, 'discount_amount' => $discount_amount, 'id' => $coupon_id);
                    return $result;
                } else {
                    $msg = "Sorry! You have used this coupon before";
                    return array('msg' => $msg);
                }
            }
        }
    }

    public function store_appointment(Request $request)
    {
        // the valdiation is in (app/requests/PatientRequest)

        $patient_id = Auth::guard('patient')->id();

        $this->validate($request, [
            'calander_date_day' => ['required'],
            'branch_id' => ['required', 'exists:branches,id'],
            'unit_id' => ['required', 'exists:units,id'],
            'service_id' => ['required', 'exists:service_items,id'],
            'coupon_id' => ['nullable', 'exists:coupons,id', new CouponRule($patient_id)],
            'appointment_note' => ['nullable', 'max:200'],
        ]);

        $patient_id = $request->input('search_patient_id');

        $calander_date_day = $request->input('calander_date_day');

        $calander_date_start = date("H:i", strtotime($request->input('calander_date_start')));

        //for only selecting start slot without end slot
        if ($request->input('calander_date_end')) {
            $calander_date_end = date("H:i", strtotime($request->input('calander_date_end')));
        } else {
            $duration = prox_sett('timeslotduration');
            $cleanup = prox_sett('timeslotcleanup');
            $calander_date_start_end =  $duration + $cleanup;
            $calander_date_end = Carbon::parse($calander_date_start)->addMinutes($calander_date_start_end);
            $calander_date_end = Carbon::parse($calander_date_start)->addMinutes($calander_date_start_end)->format("H:i");
        }

        $start_at = $calander_date_day . ' ' . $calander_date_start;
        $end_at = $calander_date_day . ' ' . $calander_date_end;

        $service_id = $request->input('service_id');
        $branch_id = $request->input('branch_id');
        $unit_id = $request->input('unit_id');
        $coupon_id = $request->input('coupon_id');

        $service_price = Service_item::select('id', 'name', 'service_inv_cat_id', 'price')->where('id', $service_id)->where('deactivate', 0)->first();

        if (!empty($coupon_id)) {
            $coupon = Coupon::find($coupon_id);
            $discount_amount = $coupon->discount($service_price->price);
            $final_price = $service_price->price - $discount_amount;
        } else {
            $discount_amount = null;
            $final_price = $service_price->price;
        }

        $appointment = Appointment::create([
            'code' => "AP" . $this->generateRandomString(6),
            'specialty_id' => $request->input('specialty_id'),
            'branch_id' => $branch_id,
            'unit_id' => $unit_id,
            'patient_id' => $patient_id,
            'services_cat_id' => $service_id,
            'start_at' => $start_at,
            'end_at' => $end_at,
            'note' => $request->input('appointment_note'),
        ]);

        $invoice = Invoice::create([
            'code' => "IN" . $this->generateRandomString(6),
            'service_inv_cat_id' => $service_price->service_inv_cat_id,
            'specialty_id' => $request->input('specialty_id'),
            'receivable_id' => $patient_id,
            'receivable_type' => "App\Models\Patient\Patient",
            'branch_id' => $branch_id,
            'items_price' => $service_price->price,
            'final_price' => $final_price,
            'coupon_id' => $coupon_id,
            'discount' => $discount_amount,
        ]);

        $patient = Patient::select('id', 'type', 'first_name', 'phone_number')->find($patient_id);
        $patient->type = 2;
        $patient->save();

        $invoice_item = Invoice_item::create([
            'invoice_id' => $invoice->id,
            'itemable_id' => $appointment->id,
            'itemable_type' => "App\Models\Branch\Appointment",
            'categorizable_id' => $service_id,
            'categorizable_type' => "App\Models\Patient\Service_item",
            'price' => $service_price->price,
            'sold_price' => $final_price,
        ]);

        session()->flash('success', 'The appointment has been created successfully');
        return redirect()->route('patient_auth.profile');
    }

    public function generateRandomString($length = 20)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    public function about()
    {
        return view('landing/about');
    }

    public function contact()
    {
        return view('landing/contact');
    }

    public function blogs()
    {
        return view('landing/blogs');
    }


    public function show_pat_appointment_public($code)
    {

        $appointment = Appointment::select('id', 'code', 'patient_id', 'branch_id', 'doctor_id', 'services_cat_id', 'creator_id', 'status', 'start_at', 'end_at')
            ->with(['patient' => function ($q) {
                $q->select('id', 'phone_number', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['branch' => function ($q) {
                $q->select('id', 'name', 'address');
            }])
            ->with(['creator' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])

            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type', 'price')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'final_price');
                    }]);
            }])

            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->where('code', $code)
            ->first();

        return view('landing/appointment_public', compact('appointment'));
    }


    public function rate_appo_public($code)
    {

        $appointment = Appointment::select('id', 'code', 'patient_id', 'branch_id', 'doctor_id', 'services_cat_id', 'creator_id', 'status', 'start_at', 'end_at')
            ->with(['patient' => function ($q) {
                $q->select('id', 'phone_number', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['branch' => function ($q) {
                $q->select('id', 'name', 'address');
            }])
            ->with(['doctor' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['creator' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->doesnthave('rate')
            ->where('code', $code)
            ->first();

        if ($appointment) {
            $status = 1;
        } else {
            $status = 2;
        }

        $cancel_reasons = Cancel_reason_cat::all();

        return view('landing/empty/rate_appo_public', compact('appointment', 'status', 'cancel_reasons'));
    }


    public function rate_appo_public_store(Request $request)
    {
        $this->validate($request, [
            'rate_type' => Rule::in([1, 2]),
            'service' => Rule::in([1, 2, 3, 4, 5]),
            'doctor' => Rule::in([1, 2, 3, 4, 5]),
            'time' => Rule::in([1, 2, 3, 4, 5]),
            'cleanliness' => Rule::in([1, 2, 3, 4, 5]),
            'appointment_code_input' => 'required|exists:appointments,code',
            'cancel_cat_id' => 'sometimes|required|exists:cancel_reason_cats,id',
            'note' => 'max:255',
        ]);

        $appointment = Appointment::select('id', 'start_at')->where('code', $request->input('appointment_code_input'))->get();


        if ($request->input('rate_type') == 1) {

            $rate = Rate_appointment::create([
                'appointment_id' => $appointment[0]->id,
                'type' => 1,
                'service' => $request->input('rate_service'),
                'doctor' => $request->input('rate_doctor'),
                'reception' => $request->input('rate_reception'),
                'time' => $request->input('rate_time'),
                'cleanliness' => $request->input('rate_cleanliness'),
                'note' => $request->input('rate_note'),
                'appointment_date' => $appointment[0]->start_at,
            ]);
        } else {
            $rate = Rate_appointment::create([
                'appointment_id' => $appointment[0]->id,
                'type' => 2,
                'cancel_cat_id' => $request->input('reason'),
                'note' => $request->input('rate_note'),
                'appointment_date' => $appointment[0]->start_at,
            ]);
        }

        return redirect()->back()->with('success', 'The rate has been created successfully');
    }
}
