<?php

namespace App\Http\Requests\School;

use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\Student;
use App\Models\SchoolSemester;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;

class StudentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules(Request $request)
    {
        if ($this->isMethod('post')) {
            return [
                'name' => 'required|string|max:100',
                'ssn' => [
                    'required',
                    'max:10',
                    function ($attribute, $value, $fail) {
                        // Check if the SSN exists in the school_employees table
                        $existsInEmployees = SchoolEmployee::where('ssn', $value)->exists();

                        // Check if the SSN exists in the students table
                        $existsInStudents = Student::where('ssn', $value)->exists();

                        if ($existsInEmployees || $existsInStudents) {
                            // If the SSN is found in either table, fail validation
                            $fail(trans('validation.custom.ssn.unique'));
                        }
                    }
                ],
                'nationality' => 'required|string|max:100',
                'school_id' => 'required|exists:schools,id',
                'class_id' => 'required|exists:school_classes,id',
            ];
        }

        if ($this->isMethod('put') || $this->isMethod('patch')) {
            $id = $this->route('id'); // Get the current record ID

            return [
                'name' => 'sometimes|string|max:100',
                'ssn' => [
                    'string',
                    'max:10',
                    function ($attribute, $value, $fail) use ($id) {
                        // Ensure the SSN is unique across all other records, but not unique for the current ones
                        $exists = SchoolEmployee::where('ssn', $value)
                            ->whereNot('id', $id) // Exclude current and related records
                            ->exists();

                        $existsSchoolEmployee = SchoolEmployee::where('ssn', $value)->exists();

                        if ($exists || $existsSchoolEmployee) {
                            // If a duplicate is found in other records, fail validation
                            $fail(trans('validation.custom.ssn.unique'));
                        }
                    },
                ],
                'nationality' => 'required|string|max:100',
                'school_id' => 'required|exists:schools,id',
                'class_id' => 'required|exists:school_classes,id',
            ];
        }

    }

}
