<?php

namespace App\Http\Requests\School;

use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\Student;
use Illuminate\Http\Request;
use Illuminate\Foundation\Http\FormRequest;

use App\Models\User;

use Illuminate\Validation\Rule;

class SchoolEmployeeRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules(Request $request)
    {
        if ($this->isMethod('post')) {
            return [
                'name' => 'required|string|max:100',
                'phone' => [
                    'required',
                    'string',
                    'regex:/^05\d{8}$/',
                    function ($attribute, $value, $fail) {
                        // Check if the Phone exists in the school_employees table
                        $existsInEmployees = SchoolEmployee::where('ssn', $value)->exists();

                        // Check if the Phone exists in the users table
                        $existsInUsers = User::where('phone', $value)->exists();

                        if ($existsInEmployees || $existsInUsers) {
                            // If the Phone is found in either table, fail validation
                            $fail(trans('validation.custom.phone.unique'));
                        }
                    }
                ],

                'ssn' => [
                    'required',
                    'max:10',
                    function ($attribute, $value, $fail) {
                        // Check if the SSN exists in the school_employees table
                        $existsInEmployees = SchoolEmployee::where('ssn', $value)->exists();

                        // Check if the SSN exists in the students table
                        $existsInStudents = Student::where('ssn', $value)->exists();

                        if ($existsInEmployees || $existsInStudents) {
                            // If the SSN is found in either table, fail validation
                            $fail(trans('validation.custom.ssn.unique'));
                        }
                    }
                ],


                'email' => 'required|string|email:dns|max:255|unique:school_employees',
                'current_work_id'=>'exists:current_works,id',
                'specialization_id'=>'required|exists:specializations,id',
                'school_id'=>'required|exists:schools,id',
            ];
        }

        if ($this->isMethod('put') || $this->isMethod('patch')) {
            $id = $this->route('id'); // Get the current record ID

            return [
                'name' => 'sometimes|string|max:100',
                'email' => [
                    'string',
                    'email:dns',
                    'max:255',
                    function ($attribute, $value, $fail) use ($id) {
                        $record = SchoolEmployee::find($id);
                        $relatedIds = SchoolEmployee::where('id','!=',$id)->where('email', $record->email)->pluck('id')->toArray();

                        // Ensure the email is unique across all other records, but not unique for the current and related ones
                        $exists = SchoolEmployee::where('email', $value)
                            ->whereNotIn('id', array_merge([$id], $relatedIds)) // Exclude current and related records
                            ->exists();
                        if ($exists) {
                            // If a duplicate is found in other records, fail validation
                            $fail(trans('validation.custom.email.unique'));
                        }
                    },
                ],
                'phone' => [
                    'string',
                    'regex:/^05\d{8}$/',
                    function ($attribute, $value, $fail) use ($id) {
                        $record = SchoolEmployee::find($id);
                        $relatedIds = SchoolEmployee::where('id','!=',$id)->where('phone', $record->phone)->pluck('id')->toArray();


                        // Ensure the phone is unique across all other records, but not unique for the current and related ones
                        $exists = SchoolEmployee::where('phone', $value)
                            ->whereNotIn('id', array_merge([$id], $relatedIds)) // Exclude current and related records
                            ->exists();

                        if ($exists) {
                            // If a duplicate is found in other records, fail validation
                            $fail(trans('validation.custom.phone.unique'));
                        }
                    },
                ],
                'ssn' => [
                    'string',
                    'max:10',
                    function ($attribute, $value, $fail) use ($id) {
                        $record = SchoolEmployee::find($id);
                        $relatedIds = SchoolEmployee::where('id','!=',$id)->where('ssn', $record->ssn)->pluck('id')->toArray();
                        // Ensure the SSN is unique across all other records, but not unique for the current and related ones
                        $exists = SchoolEmployee::where('ssn', $value)
                            ->whereNotIn('id', array_merge([$id], $relatedIds)) // Exclude current and related records
                            ->exists();

                        $existsStudent = Student::where('ssn', $value)->exists();

                        if ($exists || $existsStudent) {
                            // If a duplicate is found in other records, fail validation
                            $fail(trans('validation.custom.ssn.unique'));
                        }
                    },
                ],
                'current_work_id' => 'sometimes|exists:current_works,id',
                'specialization_id' => 'sometimes|exists:specializations,id',
                'school_id' => 'sometimes|exists:schools,id',
            ];
        }

        return [];
    }

}
