<?php

namespace App\Http\Controllers\Api\TeacherDashboard\StudentAttendance;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\AbsenseAndLateness\StudentAttendanceResource;
use Illuminate\Http\Request;
use App\Models\AdminDashboard\School\Period;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\Student;
use App\Models\AdminDashboard\School\Subject;
use App\Models\ManagerDashboard\AbsenseAndLateness\PeriodStudentAttendance;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAttendance;
use App\Models\ManagerDashboard\Tables\WaitingPeriodsSchedule\WaitingPeriodsSchedule;
use App\Models\ManagerDashboard\TeacherSchedule;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Resources\Json\ResourceCollection;

class TeacherStudentAttendanceController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-student-attendance']);
    }

    public function getDatePeriods(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }


        $schoolEmployee = SchoolEmployee::where('school_id', $request->school_id)->where('user_id', $user->id)->where('type', 'teacher')->first();
        if (!$schoolEmployee) {
            return response()->json(['error' => trans('api.you are not registered as school employee')], 404);
        }

        // Pagination
        $perPage = $request->per_page ?? 15;

        if ($request->date) {
            if ($request->date_type == "hijri") {
                $date = Helper::getCoptic($request->date);
            } else {
                try {
                    $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
                } catch (\Exception $e) {
                    return response()->json(['error' => 'Invalid date format. Please use d/m/Y.'], 400);
                }
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $semesterWeekDay = DB::table('semester_week_days')->where('date', $date)->where('semester_id', $semester->id)->first();
        if (!$semesterWeekDay) {
            return response()->json(['error' => trans('api.this day is not a study day')], 404);
        }
        $schoolStudentsCount = Student::where('school_id', $request->school_id)->count();

        $teacherSchedulePeriods = TeacherSchedule::join('teacher_schedul_data', 'teacher_schedul_data.teacher_schedule_id', 'teacher_schedules.id')
            ->select(
                // 'teacher_schedul_data.id',
                'teacher_schedul_data.school_employee_id as current_employee_id',
                'school_class_id',
                'subject_id',
                'period_id'
            )
            ->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('day_id', $semesterWeekDay->day_id)
            ->where('teacher_schedul_data.school_employee_id', $schoolEmployee->id)
            ->paginate($perPage);

        $waitingPeriodsSchedulePeriods = WaitingPeriodsSchedule::join('waiting_periods_schedule_data', 'waiting_periods_schedule_data.schedule_id', 'waiting_periods_schedules.id')
            ->select(
                // 'waiting_periods_schedule_data.id',
                'waiting_periods_schedules.school_employee_id as current_employee_id',
                'waiting_periods_schedule_data.school_employee_id as alternative_employee_id',
                'school_class_id',
                'subject_id',
                'period_id'
            )
            ->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('day_id', $semesterWeekDay->day_id)
            ->where('waiting_periods_schedule_data.school_employee_id', $schoolEmployee->id)
            ->where('creation_date', $date)
            ->paginate($perPage);

        // Merge the two paginated collections into one
        $mergedResults = collect($teacherSchedulePeriods->items())->merge($waitingPeriodsSchedulePeriods->items());

        // Replace the items in the first paginated collection with the merged items
        $teacherSchedulePeriods->setCollection($mergedResults);

        // Group periods by class_name_id
        $groupedPeriods = [];

        $teacherSchedulePeriods->each(function ($item) use ($request, $date, &$groupedPeriods, $semester) {
            $SchoolStudentsCount = PeriodStudentAttendance::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('period_id', $item->period_id)
                ->where('date', $date)
                ->count();

            $attendedSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('period_id', $item->period_id)
                ->where('status', 'attended')
                ->where('date', $date)
                ->count();

            $absentSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('date', $date)
                ->where('period_id', $item->period_id)
                ->where(function ($q) {
                    $q->where('status', 'absent')
                        ->orWhere('status', 'absent_with_excuse');
                })->count();

            $status = $SchoolStudentsCount > 0 && ($SchoolStudentsCount == $attendedSchoolStudents + $absentSchoolStudents);

            $class = SchoolClass::findOrFail($item->school_class_id);
            $currentEmployee = SchoolEmployee::find($item->current_employee_id);
            $alternativeEmployee = SchoolEmployee::find($item->alternative_employee_id);

            $periodDetails = [
                'school_class_id' => $item->school_class_id,
                'current_employee_id' => $currentEmployee->id,
                'current_employee_name' => $currentEmployee->name,
                'alternative_employee_id' => $alternativeEmployee?->id,
                'alternative_employee_name' => $alternativeEmployee?->name,
                'class_number' => $class->class_number,
                'period_id' => $item->period_id,
                'period_name' => Period::findOrFail($item->period_id)?->name,
                'subject_id' => $item->subject_id,
                'subject_name' => Subject::findOrFail($item->subject_id)?->name,
                'status' => $status ? trans('api.attended') : trans('api.not_attended'),
                'type' => isset($item->alternative_employee_id) ? 'waiting' : 'none',
            ];

            if (!isset($groupedPeriods[$class->class_name_id])) {
                $groupedPeriods[$class->class_name_id] = [
                    'class_name_id' => $class->class_name_id,
                    'class_name' => $class->className->name,
                    'date_options' => Helper::getDateOptions($date, 'coptic'),
                    'ClassRegisteredClassNumbers' => 0,
                    'AllClassNumbers' => 0,
                    'details' => []
                ];
            }

            $groupedPeriods[$class->class_name_id]['details'][] = $periodDetails;
        });


        // Convert grouped periods to collection for pagination
        $waitingPeriodsSchedulePeriods->setCollection(collect(array_values($groupedPeriods)));
        $finalPeriods = $waitingPeriodsSchedulePeriods;

        $finalPeriods = $finalPeriods->map(function ($record) {
            $ClassRegisteredClassNumbers = 0;
            foreach ($record['details'] as $period) {
                $period['status'] == trans('api.attended') ? $ClassRegisteredClassNumbers + 1 : $ClassRegisteredClassNumbers;
            }
            $record['ClassRegisteredClassNumbers'] = $ClassRegisteredClassNumbers;
            $record['AllClassNumbers'] = count($record['details']);
            return $record;
        });

        // All Top Statistics
        $attendedSchoolStudents = StudentAttendance::where('school_id', $request->school_id)->where('semester_id', $semester->id)->where('status', 'attended')->where('date', $date)->count();
        $absentSchoolStudents = StudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('semester_id', $semester->id)
            ->where(function ($q) {
                $q->where('status', 'absent')
                    ->orWhere('status', 'absent_with_excuse');
            })->count();
        $StudentsNotRegisteredYet = $schoolStudentsCount - ($attendedSchoolStudents + $absentSchoolStudents);
        $StudentsNotRegisteredYet = $StudentsNotRegisteredYet > 0 ? $StudentsNotRegisteredYet : 0;
        $RegisteredClasses = StudentAttendance::select('class_id')->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('date', $date)
            ->whereNotNull('status')
            ->groupBy('class_id')
            ->get();
        $AllClasses = Student::where('school_id', $request->school_id)->distinct('class_id')->count();

        return response()->json([
            'datePeriods' => $finalPeriods,
            'schoolStudentsCount' => $schoolStudentsCount,
            'attendedSchoolStudentsCount' => $attendedSchoolStudents,
            'absentSchoolStudentsCount' => $absentSchoolStudents,
            'StudentsNotRegisteredYetCount' => $StudentsNotRegisteredYet,
            'RegisteredClassesCount' => count($RegisteredClasses),
            'AllClassesCount' => $AllClasses,
            'message' => trans(
                'api.list of date available periods'
            )
        ]);
    }


    public function getDayPeriodStudents(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'period_id' => 'required|exists:periods,id',
            'class_name_id' => 'required|exists:class_names,id',
            'class_number' => 'required',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $class = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->where('class_number', $request->class_number)->first();
        if (!$class) {
            return response()->json(['error' => trans('api.there is no class registered for this data')], 404);
        }

        $studentIds = Student::where('school_id', $request->school_id)->where('class_id', $class->id);

        if ($request->student_name) {
            $studentIds->where('name', 'LIKE', "{$request->student_name}%");
        }

        $studentIds = $studentIds->pluck('id')->toArray();

        $checkIfCreated = PeriodStudentAttendance::where('date', $date)->where('semester_id', $semester->id)->where('class_id', $class->id)->where('period_id', $request->period_id)->where('school_id', $request->school_id)->exists();
        if (!$checkIfCreated) {
            foreach ($studentIds as $student_id) {
                $studentAttendance = new PeriodStudentAttendance();
                $studentAttendance->school_id = $request->school_id;
                $studentAttendance->semester_id = $semester->id;
                $studentAttendance->student_id = $student_id;
                $studentAttendance->class_id = Student::findOrFail($student_id)?->class_id;
                $studentAttendance->period_id = $request->period_id;
                $studentAttendance->status = 'attended';
                $studentAttendance->date = $date;
                $studentAttendance->date_type = 'coptic';
                $studentAttendance->save();
            }
        }

        // Paginate results with a default value if not provided
        $perPage = $request->per_page ?? 15;

        $studentAttendance = PeriodStudentAttendance::whereIn('student_id', $studentIds)
            ->where('class_id', $class->id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where('school_id', $request->school_id)
            ->where('date', 'LIKE', $date)
            ->paginate($perPage);
        $studentAttendances = StudentAttendanceResource::collection($studentAttendance);

        // All Top Statistics
        $schoolStudentsCount = count($studentIds);
        $attendedSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where('status', 'attended')->where('date', $date)
            ->count();
        $absentSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where(function ($q) {
                $q->where('status', 'absent')
                    ->orWhere('status', 'absent_with_excuse');
            })->count();
        $StudentsNotRegisteredYet = $schoolStudentsCount - ($attendedSchoolStudents + $absentSchoolStudents);
        $StudentsNotRegisteredYet = $StudentsNotRegisteredYet > 0 ? $StudentsNotRegisteredYet : 0;
        $RegisteredClasses = PeriodStudentAttendance::select('class_id')
            ->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where('class_id', $class->id)
            ->where('date', $date)
            ->whereNotNull('status')
            ->groupBy('class_id')
            ->get();
        $AllClasses = SchoolClass::where('school_id', $request->school_id)->count();

        return response()->json([
            'studentAttendances' => $studentAttendances,
            'schoolStudentsCount' => $schoolStudentsCount,
            'attendedSchoolStudentsCount' => $attendedSchoolStudents,
            'absentSchoolStudentsCount' => $absentSchoolStudents,
            'StudentsNotRegisteredYetCount' => $StudentsNotRegisteredYet,
            'RegisteredClassesCount' => count($RegisteredClasses),
            'AllClassesCount' => $AllClasses,
            'message' => trans(
                'api.list of students attendance'
            )
        ]);
    }

    public function studentAutocompleteName(Request $request)
    {
        $results = Student::where('school_id', $request->id)->where('name', 'LIKE', "{$request->student_name}%")
            ->select('id', 'name')
            ->get();
        return response()->json($results);
    }

    /** change status of attendance to particular employee */
    public function changePeriodAttendanceStatus(Request $request)
    {
        // Validate incoming request
        $rules = [
            'school_id' => 'required|exists:schools,id',
            'period_id' => 'required|exists:periods,id',
            'class_name_id' => 'required|exists:class_names,id',
            'class_number' => 'required',
            'date_type' => 'nullable|in:coptic,hijri',
            'records' => 'required|array|min:1', // Ensures records is an array and has at least 1 entry
            'records.*.student_id' => 'required|exists:school_students,id',
            'records.*.status' => 'required|in:attended,absent,absent_with_excuse',
        ];

        $request->validate($rules);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = $request->date;
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $class = SchoolClass::where('school_id', $request->school_id)
            ->where('class_name_id', $request->class_name_id)
            ->where('class_number', $request->class_number)
            ->first();

        if (!$class) {
            return response()->json(['error' => trans('api.class not found')], 404);
        }

        foreach ($request->input('records') as $record) {
            $studentAttendance = PeriodStudentAttendance::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('class_id', $class->id)
                ->where('period_id', $request->period_id)
                ->where('student_id', $record['student_id'])
                ->where('date', 'LIKE', "%$date%")
                ->firstOrFail();

            $studentAttendance->update([
                'status' => $record['status'],
                'absence_reason' => $record['absence_reason'] ?? null,
            ]);
        }

        return response()->json(['message' => trans('api.student attendance updated successfully')]);
    }

    /** تغييب الكل */
    public function makeAllPeriodStudentsAbsent(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'class_name_id' => 'required|exists:class_names,id',
            'class_number' => 'required',
            'period_id' => 'required|exists:periods,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $class = SchoolClass::where('school_id', $request->school_id)
            ->where('class_name_id', $request->class_name_id)
            ->where('class_number', $request->class_number)
            ->first();

        if (!$class) {
            return response()->json(['error' => trans('api.class not found')], 404);
        }

        $studentAttendances = PeriodStudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('class_id', $class->id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->get();

        foreach ($studentAttendances as $studentAttendance) {
            $studentAttendance->update([
                'status' => 'absent',
                'absence_reason' => null,
            ]);
        }
        return response()->json(['message' => trans('api.all student Attendance updated to Absent Successfully')]);
    }
}
