<?php

namespace App\Http\Controllers\Api\ManagerDashboard\TeamAndCommitteeMeetings;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;

use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\CommitteeRequest;
use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\UpdateCommiteeRequest;
use Illuminate\Http\Request;
use App\Http\Resources\ManagerDashboard\TeamAndCommitteeMeetings\CommitteeResource;
use App\Models\AdminDashboard\School\CurrentWork;
use App\Models\AdminDashboard\School\School;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Committee;
use App\Traits\LogsActivity;

class CommitteeController extends Controller
{
    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-semester-vacations']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request, School $school)
    {
        // Start with the base query
        $data = Committee::with(['meetings' => function ($query) use ($school) {
            // Filter meetings by school_id
            $query->where('school_id', $school->id);
        }]);
        // Retrieve all query parameters
        $queryParams = $request->query();


        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Committee', $queryParams);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;


        $filterData = CommitteeResource::collection($data->where('school_id', Null)->orWhere('school_id', $school->id)->paginate($perPage));


        return response()->json([
            'commitees' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById(Committee $committee, School $school)
    {
        // Define mapping of assignment form numbers to assignment works and work IDs
        $assignmentData = [
            30 => [
                'assignment_works' => ['رئيساً للجنة', 'مقرر للجنة', 'عضو', 'عضو', 'عضو', 'عضو'],
                'work_ids' => [6, 11, 15, 2, 2, 2]
            ],
            31 => [
                'assignment_works' => ['رئيساً للجنة', 'عضو', 'عضو', 'عضو', 'عضو', 'مقرر للجنة', 'عضو', 'عضو', 'عضو'],
                'work_ids' => [1, 6, 7, 8, 11, 15, 2, 2, 2]
            ],
            32 => [
                'assignment_works' => ['رئيساً للجنة', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'مقرر للجنة'],
                'work_ids' => [1, 6, 7, 8, 11, 15, 2, 2, 2, 9]
            ],
            33 => [
                'assignment_works' => ['رئيساً للجنة', 'عضو', 'عضو', 'عضو', 'عضو', 'مقرر للجنة'],
                'work_ids' => [7, 6, 2, 2, 2, 11]
            ],
            36 => [
                'assignment_works' => ['رئيساً للجنة', 'نائب الرئيس', 'عضو ومقرر اللجنة', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو'],
                'work_ids' => [1, 7, 10, 11, 15, 2, 2, 2, 2, 2]
            ],
            183 => [
                'assignment_works' => ['رئيساً للجنة', 'نائب الرئيس', 'عضو ومقرر', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو'],
                'work_ids' => [1, 6, 11, 2, 2, 2, 2, 2, 2]
            ],
        ];

        // Default assignment works if no match is found
        $defaultAssignmentWorks = ['رئيساً للجنة', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو'];

        // Get assignment works and work IDs for the specific form number or use default values
        $assignmentWorks = $assignmentData[$committee->assignment_form_number]['assignment_works'] ?? $defaultAssignmentWorks;
        $currentWorkIds = $assignmentData[$committee->assignment_form_number]['work_ids'] ?? [];

        // Fetch current works based on form number
        if (in_array($committee->assignment_form_number, [35, null])) {
            $currentWorks = [];
        } else {
            // Fetch the works based on the work_ids
            $currentWorks = CurrentWork::with(['employees' => function ($query) use ($school) {
                $query->where('school_id', $school->id);
            }])
                ->select('id', 'job_title')
                ->whereIn('id', $currentWorkIds)
                ->get();

            // Reorder and duplicate the works based on the order in work_ids
            $currentWorks = collect($currentWorkIds)->map(function ($workId) use ($currentWorks) {
                return $currentWorks->firstWhere('id', $workId);
            });
        }


        // Define the default work ids to fetch employees if current work has no employees
        $defaultWorkIds = [1, 6, 7, 8];

        // Loop through currentWorks and check if any work has no employees
        if ($currentWorks) {
            $currentWorks->each(function ($work) use ($defaultWorkIds) {
                if ($work->employees->isEmpty()) {
                    // If no employees for the current work, fetch from defaultWorkIds
                    $defaultWork = CurrentWork::with(['employees' => function ($query) {
                        $query->where('school_id', auth()->user()->school_id);
                    }])
                        ->select('id', 'job_title')
                        ->whereIn('id', $defaultWorkIds)
                        ->first();

                    // Replace empty employees with the fetched employees from the default work
                    if ($defaultWork && !$work->employees->isNotEmpty()) {
                        $work->employees = $defaultWork->employees;
                    }
                }
            });
        }

        // Load meetings for the committee
        $committee->load(['meetings' => function ($query) use ($school) {
            $query->where('school_id', $school->id);
        }]);

        // Pass current work, assignment works, and flag to CommitteeResource
        $committeeResource = new CommitteeResource($committee, $currentWorks, $assignmentWorks, true);

        return response()->json([
            'committee' => $committeeResource,
            'message' => trans('api.committee details')
        ], 200);
    }


    /* this method will stord data to DB Model*/
    public function store(CommitteeRequest $request)
    {

        $data = $request->validated();
        $committee = new Committee();
        $committee->school_id =  $data['school_id'];
        $committee->name = $data['name'];
        $committee->form_number = $data['form_number'] ?? 35;
        $committee->assignment_form_number = $data['assignment_form_number'] ?? 35;
        $committee->status =  $data['status'] ?? 1;
        $committee->save();
        // ------------------------------- School ID -----------------------
        $school_id = $request->school_id;
        $this->ManagerlogActivity('Committee', 'store', $school_id, $data['assignment_form_number'] ?? 35);

        return response()->json(['message' => trans('api.record stored successflly')], 200);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(UpdateCommiteeRequest $request, Committee $committee, School $school)
    {
        $data = $request->validated();
        $committee->update($data);
        // ------------------------------- School ID -----------------------
        $school_id = $school->id;
        $this->ManagerlogActivity('Committee', 'update', $school_id,  $committee->form_number);


        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    /* this method will change activation status for
       specific Model recored by using ID
    */
    public function changeStatus(Request $request, Committee $committee, School $school)
    {
        $committee->status = $request->status;
        $committee->save();

        return response()->json(['message' => trans('api.status changed successflly')], 200);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive(Committee $committee, School $school)
    {
        $school_id = $committee->school_id;
        $committee->delete();
        // ------------------------------- School ID -----------------------
        $this->ManagerlogActivity('Committee', 'archive', $school_id);

        return response()->json(['message' => trans('api.record archived successflly')], 200);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function destroy(Committee $committee, School $school)
    {
        $committee->forceDelete();
        $this->ManagerlogActivity('Committee', 'delete', $school->id);
        return response()->json(['message' => trans('api.record deleted successflly')], 200);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore(Committee $committee, School $school)
    {
        $committee->restore();
        $this->ManagerlogActivity('Committee', 'restore', $school->id);
        return response()->json(['message' => trans('api.record restored successflly')], 200);
    }
}
