<?php

namespace App\Http\Controllers\Api\ManagerDashboard\TeacherSchedule;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\ManagerDashboard\TeacherSchedule\TeacherScheduleStoreRequest;
use App\Http\Requests\ManagerDashboard\TeacherSchedule\TeacherScheduleUpdateRequest;
use App\Http\Resources\ManagerDashboard\TeacherSchedule\SchoolClassResource;
use App\Models\AdminDashboard\School\Period;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\Day;
use App\Models\ManagerDashboard\TeacherSchedule;
use App\Models\ManagerDashboard\TeacherScheduleData;
use App\Traits\LogsActivity;

class TeacherScheduleController extends Controller
{
    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify','permission:teacherSchedules']);
    }


    public function getAll(Request $request, $school_id)
    {
        // Fetch class IDs for the given school
        $class_ids = SchoolClass::where('school_id', $school_id)->pluck('id')->toArray();

        // Apply filters based on query parameters, if provided
        $queryParams = $request->query();
        if (!empty($queryParams)) {
            $classes = SchoolClass::where('school_id', $school_id);
            foreach ($queryParams as $attr => $parameter) {
                $classes->where($attr, $parameter);
            }
            $class_ids = $classes->pluck('id')->toArray();
        }

        $data = TeacherSchedule::whereIn('school_class_id', $class_ids);
        $data_check = TeacherSchedule::where('school_id', $school_id)->pluck('semester_id')->unique()->toArray();
        if (count($data_check) < 3) {
            Helper::saveTeacherSchedules($request->school_id);
            $data = TeacherSchedule::whereIn('school_class_id', $class_ids);
        }

        // Paginate the results, with default per-page value of 15
        $perPage = $queryParams['per_page'] ?? 2000;
        $paginatedData = $data->orderBy('semester_id')->paginate($perPage);

        // Group the data by semester name (assuming 'semester' is a relationship in SchoolClassResource)
        $groupedData = SchoolClassResource::collection($paginatedData->getCollection())->groupBy(function ($item) {
            return $item['semester']['name']; // Group by semester name
            
        });

        // Convert the grouped data into an array of objects
        $groupedArray = [];
        foreach ($groupedData as $semester => $schedules) {
            $groupedArray[] = [
                'semester' => $semester,
                'semester_id' => $schedules->values()?->first()?->semester_id,
                'teacherSchedules' => $schedules->values(), // Reset keys after grouping
            ];
        }

        // Return the structured JSON response with pagination info
        return response()->json([
            'data' => $groupedArray,
            'message' => trans('api.list of all available records'),
        ]);
    }


    /* this method will return specific Model recored 
        by using ID
    */
    public function getById($teacher_schedule_id)
    {
        $teacherSchedule = TeacherSchedule::findOrFail($teacher_schedule_id);
        $groupedData = $teacherSchedule->schedule_data->groupBy('day_id');
        $formattedData = $groupedData->map(function ($dayData, $dayId) {
            return [
                'day_id' => $dayId,
                'day' => $dayData->first()->day_name->name,
                'data' => $dayData->map(function ($entry) {
                    return [
                        'subject_id' => $entry->subject_id,
                        'subject' => $entry->subject->name,
                        'teacher_id' => $entry->school_employee_id,
                        'teacher' => $entry->employee?->name,
                        'period_id' => $entry->period_id,
                        'period' => $entry->period->name,
                    ];
                })->values()
            ];
        })->values();
        return response()->json([
            'semester_id'=> $teacherSchedule->semester?->id,
            'semester_name'=> $teacherSchedule->semester?->name,
            'class_id' => $teacherSchedule->class?->className?->id,
            'class_name' => $teacherSchedule->class?->className?->name,
            'class_number' => $teacherSchedule->class?->class_number,
            'teacherSchedule' => $formattedData,
            'message' => trans('api.teacherScheduals details')
        ]);
    }




    /* this method will stord data to DB Model*/
    public function store(TeacherScheduleStoreRequest $request)
    {
        $teacher_schedul = TeacherSchedule::findOrFail($request->teacher_schedual_id);
        // $teacher_schedul = TeacherSchedule::create([
        //     'school_id' => $request->school_id,
        //     'semester_id' => $request->semester_id,
        //     'school_class_id' => $request->school_class_id,
        // ]);

        foreach ($request->data as $data_per_day) {
            foreach ($data_per_day['data'] as $new_record) {
                TeacherScheduleData::create([
                    'teacher_schedule_id' => $request->teacher_schedual_id,
                    'subject_id' => $new_record['subject_id'],
                    'school_employee_id' => $new_record['school_employee_id'],
                    'day_id' => $data_per_day['day_id'],
                    'period_id' => $new_record['period_id'],
                ]);
            }
        }
        // ------------------------------- School ID -----------------------
        $school_id = $teacher_schedul->school_id;
        $this->ManagerlogActivity('TeacherScheduleData', 'store', $school_id);

        return response()->json(['message' => trans('api.record stored successflly')]);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(TeacherScheduleUpdateRequest $request, $id)
    {
        $teacher_schedul = TeacherSchedule::findOrFail($id);

        TeacherScheduleData::where('teacher_schedule_id', $id)->delete();

        foreach ($request->data as $data_per_day) {
            foreach ($data_per_day['data'] as $new_record) {
                TeacherScheduleData::create([
                    'teacher_schedule_id' => $id,
                    'subject_id' => $new_record['subject_id'],
                    'school_employee_id' => $new_record['school_employee_id'],
                    'day_id' => $data_per_day['day_id'],
                    'period_id' => $new_record['period_id'],
                ]);
            }
        }
        // ------------------------------- School ID -----------------------
        $school_id = $teacher_schedul->school_id;
        $this->ManagerlogActivity('TeacherScheduleData', 'update', $school_id);

        return response()->json(['message' => trans('api.record updated successflly')]);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function delete($id)
    {
        $teacher_schedul = TeacherSchedule::findOrFail($id);
        $school_id = $teacher_schedul->school_id;
        $teacher_schedul->forceDelete();
        // ------------------------------- School ID -----------------------
        $this->ManagerlogActivity('TeacherScheduleData', 'delete', $school_id);
        return response()->json(['message' => trans('api.record deleted successflly')]);
    }


    public function dev_create_school_table(Request $request)
    {
        Helper::saveTeacherSchedules($request->school_id);
        return response()->json(['message' => trans('api.record stored successflly')]);
    }
    public function dev_create_supervisor_school_table(Request $request)
    {
        Helper::saveSupervisionSchedules($request->school_id);
        return response()->json(['message' => trans('api.record stored successflly')]);
    }



    public function get_classes_for_teacher(School $school, SchoolSemester $semester, SchoolEmployee $teacher)
    {
        $teacher_schedule_classes = TeacherSchedule::where('semester_id', $semester->id)->where('school_id', $school->id)
            ->whereHas('schedule_data', function ($query) use ($teacher) {
                $query->where('school_employee_id', $teacher->id);
            })->with('class')->get()->pluck('class')->flatten()->unique('id')
            ->map(function ($class) {
                return [
                    'school_class_id' => $class->id,
                    'class_number' => $class->class_number,
                ];
            })
            ->values();

        return response()->json($teacher_schedule_classes);
    }

    public function get_teachers_with_availability(TeacherSchedule $table)
    {
        $data=[];
        $days = Day::all();
        $periods = Period::all();
        foreach ($days as $day) {
            foreach ($periods as $period) {
                $teachers_data = TeacherScheduleData::with('teacher_schedual')
                    ->where('period_id', $period->id)
                    ->where('day_id', $day->id)
                    ->whereHas('employee', function ($query) use ($table) {
                        $query->where('school_id', $table->school_id);
                    })
                    ->whereHas('teacher_schedual', function ($query) use ($table) {
                        $query->where('semester_id', $table->semester_id);
                    })
                    ->get();
                $day_period_teachers = SchoolEmployee::where('school_id', $table->school_id)->where('type', 'teacher')->get()
                    ->map(function ($teacher) use ($teachers_data) {
                        $check_exist = $teachers_data->where('school_employee_id', $teacher->id)->first();
                        return [
                            'id' => $teacher->id,
                            'name' => $teacher->name,
                            'available' => $check_exist ? 0 : 1,
                            'school_class' => $check_exist?->teacher_schedual->class?->className->name,
                            'class_number' => $check_exist?->teacher_schedual->class?->class_number,
                        ];
                    });
                $data[]=[
                    'day_id'=> $day->id,
                    'period_id'=> $period->id,
                    'teachers' => $day_period_teachers
                ];
            }
        }

        return response()->json($data);
    }
}
