<?php

namespace App\Http\Controllers\Api\AdminDashboard\Notification;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Notification\NotificationStoreRequest;
use App\Http\Requests\Notification\NotificationUpdateRequest;
use App\Models\Notification;
use App\Http\Resources\Notification\NotificationResource;
use App\Models\AdminDashboard\School\School;
use App\Models\Event;
use App\Models\NotificationReceptor;
use App\Traits\LogsActivity;
use Carbon\Carbon;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\DB;


class NotificationController extends Controller
{
    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify', 'permission:notifications']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request)
    {
        // Start with the base query
        $data = Notification::query();

        // Retrieve all query parameters
        $queryParams = $request->query();

        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\Notification', $queryParams);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($queryParams['date_type']) && isset($queryParams['date'])) {
            $date = $queryParams['date'];

            if ($queryParams['date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['date']);
            }
            $data = $data->where('date', 'like', "%$date%");
        }

        $filterData = NotificationResource::collection($data->paginate($perPage));

        return response()->json([
            'notifications' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById($id)
    {
        $notification = Notification::findOrFail($id);
        $notification = new NotificationResource($notification);
        return response()->json(['notification' => $notification, 'message' => trans('api.page details')]);
    }

    /* this method will stord data to DB Model*/
    public function store(notificationStoreRequest $request)
    {
        DB::beginTransaction();
        try {
            $notificationData = $request->except('selected_targets');
            if (isset($notificationData['date_type']) && isset($notificationData['date'])) {
                if ($notificationData['date_type'] === 'hijri') {
                    // Convert the Hijri date to Coptic
                    $notificationData['date'] = Helper::getCoptic($notificationData['date']);

                    // Optional: Change the date_type to 'coptic' if you want to store it as such
                    $notificationData['date_type'] = 'coptic';
                }
            }else{
                $notificationData['date'] = Carbon::now()->format('d/m/Y');
            }

            $notification = Notification::create($notificationData);
            switch ($request->receptor) {
                case "all_cities":
                    $schools = School::all();
                    foreach ($schools as $school) {
                        $notification->receptors()->create([
                            'school_id' => $school->id,
                            'user_id' => $school->user_id,
                            'user_type' => 'employee',
                            'notification_id' => $notification->id,
                        ]);
                    }
                    break;
                case "specific_cities":
                    $schools = School::whereIn('city_id', $request->selected_targets)->get();

                    if (count($schools) == 0) {
                        DB::rollBack();
                        return response()->json(['message' => trans('api.no schools exist in this regions')], 404);
                    }
                    foreach ($schools as $school) {
                        $notification->receptors()->create([
                            'school_id' => $school->id,
                            'user_id' => $school->user_id,
                            'user_type' => 'employee',
                            'notification_id' => $notification->id,
                        ]);
                    }
                    
                    break;
                case "specific_receptors":
                    $schools = School::whereIn('user_id', $request->selected_targets)->get();
                    foreach ($schools as $school) {
                        $notification->receptors()->create([
                            'school_id' => $school->id,
                            'user_id' => $school->user_id,
                            'user_type' => 'employee',
                            'notification_id' => $notification->id,
                        ]);
                    }
                    break;
            }
            $this->AdminlogActivity('Notification', 'store', '/notifications/' . $notification->id);
            DB::commit();
            return response()->json(['message' => trans('api.record stored successflly')]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(notificationUpdateRequest $request, $id)
    {

        $notification = Notification::findOrFail($id);
        try {
            DB::beginTransaction();
            $notificationData = $request->except('selected_targets');
            if (isset($notificationData['date_type'])) {
                if ($notificationData['date_type'] === 'hijri') {
                    // Convert the Hijri date to Coptic
                    $notificationData['date'] = Helper::getCoptic($notificationData['date']);

                    // Optional: Change the date_type to 'coptic' if you want to store it as such
                    $notificationData['date_type'] = 'coptic';
                }
            }

            $notification->update($notificationData);
            NotificationReceptor::where('notification_id', $id)->delete();
            switch ($request->receptor) {
                case "all_cities":
                    $school_ids = School::pluck('id')->toArray();
                    foreach ($school_ids as $school_id) {
                        $notification->receptors()->create([
                            'school_id' => $school_id,
                            'notification_id' => $id,
                        ]);
                    }
                    break;
                case "specific_cities":
                    $school_ids = School::whereIn('city_id', $request->selected_targets)->pluck('id')->toArray();
                    if (count($school_ids) == 0) {
                        DB::rollBack();
                        return response()->json(['message' => trans('api.no schools exist in this regions')], 404);
                    }
                    foreach ($school_ids as $school_id) {
                        $notification->receptors()->create([
                            'school_id' => $school_id,
                            'notification_id' => $id,
                        ]);
                    }
                    break;
                case "specific_receptors":
                    foreach ($request->selected_targets as $school_id) {
                        $notification->receptors()->create([
                            'school_id' => $school_id,
                            'notification_id' => $id,
                        ]);
                    }
                    break;
            }
            $this->AdminlogActivity('Notification', 'update', '/notifications/' . $id);

            DB::commit();

            return response()->json(['message' => trans('api.record updated successflly')]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive($id)
    {
        Notification::findOrFail($id)->delete();
        $this->AdminlogActivity('Notification', 'archive');

        return response()->json(['message' => trans('api.record archived successflly')]);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function delete($id)
    {
        Notification::findOrFail($id)->forceDelete();
        $this->AdminlogActivity('Notification', 'delete');

        return response()->json(['message' => trans('api.record deleted successflly')]);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore($id)
    {
        Notification::withTrashed()->findOrFail($id)->restore();
        $this->AdminlogActivity('Notification', 'restore');

        return response()->json(['message' => trans('api.record restored successflly')]);
    }
    
   
}
